
#include <linux/autoconf.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/module.h>
#include <linux/kref.h>
#include <linux/smp_lock.h>
#include <linux/usb.h>
#include <asm/uaccess.h>
#include <linux/device.h>
#include <linux/string.h>
#include <linux/firmware.h>
#include <linux/delay.h>
#include <linux/vmalloc.h>
#include "define.h"
#include "biu_reg.h"
#include "af901x.h"
#include "error.h"

u32 ucSN=0;

static u8 RegMasks[8] =
{
    0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F, 0xFF
};

static u16 USB_SendReq(MercuryRequest *req)
{
    u32     dwError = ERR_USB_NO_ERROR;
    u16		wTemp;
    u8		ucTemp0;
    u8		ucTemp1;
    u8      packet[64];
    s32		i = 0;
    u8      ucIsRead = 0;
    s32     ret,act_len;
    ucSN++;
    //PDEBUG("USB_SendReq\n");

    //
    // Prepare request packet
    //
    //					=====================
    //					|	    Header	| byte 0
    //					+-------------------+
    //					|	 SN		| byte 1
    //					+-------------------+
    //					|   2Wire Addr  	| byte 2
    //					+-------------------+
    //					|Starting Addr (Hi)	| byte 3
    //			      	+-------------------+
    //					|Starting Addr (Lo)	| byte 4
    //			      	+-------------------+
    //					| Mail Box Command	| byte 5
    //			      	+-------------------+
    //					| Register Addr Len	| byte 6
    //			      	+-------------------+
    //					|   Data Length		| byte 7
    //			_____	+-------------------+
    //			  ^		|    DATA0		| byte 8
    //			  |		+-------------------+
    //			  |			:
    //			DATA[N]    	:
    //			  |		+-------------------+
    //			__|___	|	    DATAn-1		| byte 8 + n - 1
    //					=====================

    for (i = 0; i < 64; i++)
    {
       	packet[i] = 0;
    }

	packet[0] = req->ucCmd;

	packet[1] = req->ucSeqNo;
	// encode type
	packet[2] = req->uc2WireAddr;

	// encode starting address
	wTemp = req->wStartAddr;
	ucTemp0 = wTemp & 0x00FF;
	ucTemp1 = (wTemp & 0xFF00) >> 8;
	packet[3] = ucTemp1;
    packet[4] = ucTemp0;

    // encode mail box command
    packet[5] = req->ucMbxCmd;

    // encode register address length
    packet[6] = req->ucRegAddrBytes;

    // encode data length
    packet[7] = req->ucDataLen;

	if (req->ucpWriteDataPtr)
	{
		for (i = 0; i < req->ucDataLen; i++)
		{
			packet[8 + i] = *(req->ucpWriteDataPtr + i);
			
		}
	}
	else
	{
		if (req->ucpReadDataPtr)
		{
		    ucIsRead = 1;
	    }
	}

	if ((req->ucCmd == USB_CMD_FW_DOWNLOAD) || (req->ucCmd == USB_CMD_RE_DOWNLOAD_FW) || (req->ucCmd ==USB_CMD_RE_LINK))
	{	
		//PDEBUG("USB_1\n");
		//printk(KERN_NOTICE"USB_S1 : W_C = [%d][%d][%d][%d][%d][%d][%d][%d]\n",
		//packet[0],packet[1],packet[2],packet[3],packet[4],packet[5],packet[6],packet[7]);		
		//printk(KERN_NOTICE"usb_get_dev(udev) %d\n",usb_get_dev(udev));
		ret = usb_bulk_msg(usb_get_dev(udevs),
			usb_sndbulkpipe(usb_get_dev(udevs), 0x02),
			packet,
			8 + req->ucWriteDataLen,
			&act_len,
			1000);

	}
	else
	{
	
		//PDEBUG("USB_S2 : W_C = [%d][%d][%d][%d][%d][%d][%d][%d]\n",
		//packet[0],packet[1],packet[2],packet[3],packet[4],packet[5],packet[6],packet[7]);

		ret = usb_bulk_msg(usb_get_dev(udevs),
			usb_sndbulkpipe(usb_get_dev(udevs), 2),
			packet,
			8 + req->ucWriteDataLen,
			&act_len,
			1000);

		//printk(KERN_NOTICE"Return[%d]-Len[%d]-Act_Len[%d] \n",ret,8 + req->ucWriteDataLen,act_len);
		if (ret<0)
		{
			//printk(KERN_NOTICE"Write_Fail\n");
			goto exit;
		}
		else
		{
			u8 *      ucpRxPkt=NULL;
    		u8        RxBuf[256];
    		s32		  i,ret;
    		s32       nBytesRead;
			
			ucpRxPkt = &RxBuf[0];			

    		ret = usb_bulk_msg(usb_get_dev(udevs),
				usb_rcvbulkpipe(usb_get_dev(udevs),129),
				RxBuf,
				255,
				&nBytesRead,
				1000);

			if (ret==0)        // use > or >= ??
    		{
				//printk(KERN_NOTICE"USB_S2 : R_C = [%d][%d][%d][%d][%d][%d][%d][%d]\n",
				//RxBuf[0],RxBuf[1],RxBuf[2],RxBuf[3],RxBuf[4],RxBuf[5],RxBuf[6],RxBuf[7]);	


				if (nBytesRead == 0)
        		{
            		dwError = ERR_USB_INVALID_READ_SIZE;
	    			//printk(KERN_NOTICE"ERR_USB_INVALID_READ_SIZE\n");
            		goto exit;
        		}

        		//					=====================
        		//					|	 SN	| byte 0
        		//					+-------------------+
        		//					|   Error Code 	| byte 1
        		//		_____		+-------------------+
       			//			  ^		|    DATA0	| byte 2
    			//			  |		+-------------------+
        		//			  |		:
        		//			DATA    	:
        		//			  |		+----usb_get_dev(udev)---------------+
        		//			__|___		|    DATAn - 1   | byte 2 + n - 1
        		//					=====================

        		// check if the received packet is expected ACK

        		// check header
        		if (*ucpRxPkt != (u8)(ucSN-1))
        		{
		
            		dwError = ERR_USB_INVALID_SN + *ucpRxPkt;
	    			//printk(KERN_NOTICE"ERR_USB_INVALID_SN %d - %d\n",ucSN-1,RxBuf[0]);
            		goto exit;
        		}

        		// check status
        		if (*(ucpRxPkt + 1) != 0)
        		{
            		dwError = ERR_USB_BAD_STATUS + *(ucpRxPkt + 1);
	    			//printk(KERN_NOTICE"ERR_USB_BAD_STATUS %d - %d\n",dwError,RxBuf[1]);
            		goto exit;
        		}
	
       			//
        		// Read Data Here
        		//

        		if (ucIsRead)     // for read case
        		{
            		for (i = 0; i < req->ucReadDataLen; i++)
            		{
               			*(req->ucpReadDataPtr + i) = *(ucpRxPkt + 2 + i);
            		}
	    			//PDEBUG("Read Data OK\n");
				}

    		} // if (ReadFile)
    		else
    		{
        		dwError = ERR_USB_READFILE_FAIL;
				//printk(KERN_NOTICE"ERR_USB_READFILE_FAIL\n");
    		}
			//dwError = USB_CheckAck(udev, req->ucReadDataLen, req->ucpReadDataPtr, ucIsRead, req->ucCmd);
		}
	
	}

exit:
    return(dwError);
}

//**********************************************************************
// Send Register/Memory Read Command to USB device
//
// Return:  True if successful, false otherwise
//**********************************************************************
u16 USB_ReadRegs(
	u8           ucDemod2WireAddr,
	u8           ucSlaveDemod,
	u16          wStartPos,
	u8           ucCount,
	u8 *         ucpReadBuf	
)
{
	u16 dwError = ERR_USB_NO_ERROR;
	MercuryRequest      req;
	
	if (ucSlaveDemod)
	{
        	//dwError = USB_NoTriggerDemod2MemAccess(handle, ucDemod2WireAddr, wStartPos, ucCount, ucpReadBuf, USB_FLAG_READ);
        	//if (dwError) goto exit;
	}
    else
    {
       	req.uc2WireAddr     = ucDemod2WireAddr;
       	req.ucCmd           = USB_CMD_MEM_READ;
       	req.ucpReadDataPtr  = ucpReadBuf;
       	req.ucpWriteDataPtr = NULL;
       	req.ucReadDataLen   = ucCount;
       	req.ucSeqNo         = ucSN;
       	req.ucWriteDataLen  = 0;
       	req.wStartAddr      = wStartPos;
       	req.ucDataLen       = ucCount;

       	dwError = USB_SendReq(&req);

       	if (dwError) goto exit;
    }

exit:
	return (dwError);
}
//EXPORT_SYMBOL(USB_ReadRegs);


//**********************************************************************
// Read EEPROM function
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
u32 USB_ReadEeprom(
   	u8                ucEeprom2WireAddr,
   	u16                wStartPos,
   	u8                ucCount,
   	u8 *              ucpReadBuf
)
{
    u32           dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;
    u8            ucRegAddrLen;

    if (ucEeprom2WireAddr < 0xA8)
    {
        ucRegAddrLen = 1;
    }
    else
    {
        ucRegAddrLen = 2;
    }

    req.uc2WireAddr     = ucEeprom2WireAddr + 1;
    req.ucCmd           = USB_CMD_EEPROM_READ;
    req.ucpReadDataPtr  = ucpReadBuf;
    req.ucpWriteDataPtr = NULL;
    req.ucReadDataLen   = ucCount;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = 0;
    req.wStartAddr      = wStartPos;
	req.ucRegAddrBytes  = ucRegAddrLen;
    req.ucDataLen       = ucCount;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

exit:
    return (dwError);
}
//EXPORT_SYMBOL(USB_ReadEeprom);

//**********************************************************************
// Send Boot Command to Device
//
// Return:  0 if successful, non-zero value otherwise
//**********************************************************************
static u32 USB_Boot(u8 check)
{
	u32  dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;

    req.uc2WireAddr     = 0x3A;
    req.ucCmd           = USB_CMD_BOOT;
    req.ucpReadDataPtr  = NULL;
    req.ucpWriteDataPtr = NULL;
    req.ucReadDataLen   = 0;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = 0;
    req.wStartAddr      = 0;
    req.ucDataLen       = 0;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

exit:
    return (dwError);
}
//**********************************************************************
// Download firmware
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
static u32 USB_FwDownload(
    u16  wStartPos,
    u8   ucCount,
    u8 * ucpFwBuf
)
{
	u32 dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;

    req.uc2WireAddr     = 0x3A;
    req.ucCmd           = USB_CMD_FW_DOWNLOAD;
    req.ucpReadDataPtr  = NULL;
    req.ucpWriteDataPtr = ucpFwBuf;
    req.ucReadDataLen   = 0;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = ucCount;
    req.wStartAddr      = wStartPos;
    req.ucDataLen       = ucCount;
	req.ucRegAddrBytes  = 0;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

exit:
    return (dwError);
}
//**********************************************************************
// Download firmware function
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
u32 USB_DownloadFw(
    u8        ucDemod2WireAddr,
    u16       dwArrayCnt,
    u8 *      ucpBufPtr
)
{
    u32           dwError = ERR_USB_NO_ERROR;
    u32           i;
    u32            ucConfig;
    u16           wStartAddr;
    u8 *          ucpTempPtr;
    u32           dwBlocks;
    u32           dwRemainder;

    //printk(KERN_NOTICE"Enter USB_DownloadFw\n");

	dwError = USB_GetCurrConfig(&ucConfig);
    if (dwError) goto exit;

    if (ucConfig != 1)
    {
        dwError = ERR_USB_BOOT_BAD_CONFIG_VALUE;
        goto exit;
    }
    
    dwBlocks    = (dwArrayCnt) / 63;
    dwRemainder = (dwArrayCnt) % 63;

    ucpTempPtr = ucpBufPtr;
    wStartAddr = (u16)((*(ucpTempPtr + 3) << 8) + *(ucpTempPtr + 4));

    for (i = 0; i < dwBlocks; i++)
    {            
        dwError = USB_FwDownload(wStartAddr, *(ucpTempPtr + 7), ucpTempPtr + 8);
        if (dwError) goto exit;

        ucpTempPtr += 63;
        wStartAddr = (u16)((*(ucpTempPtr + 3) << 8) + *(ucpTempPtr + 4));
    }

    if (dwRemainder)
    {
        //printk(KERN_NOTICE"enter dwRemainder USB_FwDownload\n");
        wStartAddr = (u16)((*(ucpTempPtr + 3) << 8) + *(ucpTempPtr + 4));
        dwError = USB_FwDownload(wStartAddr, *(ucpTempPtr + 7), ucpTempPtr + 8);
        if (dwError) {
            //printk(KERN_NOTICE"USB_FwDownload Fail\n");
            goto exit;
        }
    }
    
    dwError = USB_Boot(1);
    if (dwError) {
        //printk(KERN_NOTICE"USB_Boot Fail\n");
        goto exit;
    }
    
    // Wait for booting    
    mdelay(1000);
    
    dwError = USB_GetCurrConfig(&ucConfig);
    if (dwError) {
        //printk(KERN_NOTICE"USB_GetCurrConfig Fail\n");
        goto exit;
    }

    if (ucConfig != 2)
    {
        dwError = ERR_USB_BOOT_BAD_CONFIG_VALUE;
		//printk(KERN_NOTICE"ERR_USB_BOOT_BAD_CONFIG_VALUE\n");
        goto exit;
    }

exit:
    return(dwError);
}



int USB_GetCurrConfig(u32* ucpTemp)
{
    u16           dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;
    u8            ucTemp;    

    req.uc2WireAddr     = 0x3A;
    req.ucCmd           = USB_CMD_GET_CURR_CONFIG;
    req.ucpReadDataPtr  = &ucTemp;
    req.ucpWriteDataPtr = NULL;
    req.ucReadDataLen   = 1;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = 0;
    req.wStartAddr      = 0;
    req.ucDataLen       = 1;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

    *ucpTemp = ucTemp;

exit:
    return (dwError);

}



//**********************************************************************
// Write Command to Demod
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
u32 USB_WriteVirtualRegs(
    u16                wStartPos,
    u8                 ucCount,
    u8 *               ucpBuf
)
{
    u32           dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;

    req.uc2WireAddr     = 0x3A;
    req.ucCmd           = USB_CMD_VIRTUAL_MEM_WRITE;
    req.ucpReadDataPtr  = NULL;
    req.ucpWriteDataPtr = ucpBuf;
    req.ucReadDataLen   = 0;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = ucCount;
    req.wStartAddr      = wStartPos;
    req.ucDataLen       = ucCount;
    req.ucRegAddrBytes  = 0;            // only support can tuner now

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

exit:
   return (dwError);
}
//**********************************************************************
// Write register/memory function
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
u32 USB_WriteRegs(
    u8                   ucDemod2WireAddr,
    u8                   ucSlaveDemod,
   	u16                  wStartPos,
    u8                   ucCount,
    u8 *                 ucpWriteBuf
)
{
    u32              dwError = ERR_USB_NO_ERROR;
    MercuryRequest      req;


   if (((wStartPos & 0xFF00) == 0xFF00) ||
		((wStartPos & 0xAE00) == 0xAE00))
	{
			dwError = USB_WriteVirtualRegs(wStartPos, ucCount, ucpWriteBuf);
			if (dwError) goto exit;
	}
	else
	{		
        req.uc2WireAddr     = 0x3A;
        req.ucCmd           = USB_CMD_MEM_WRITE;
        req.ucpReadDataPtr  = NULL;
        req.ucpWriteDataPtr = ucpWriteBuf;
        req.ucReadDataLen   = 0;
        req.ucSeqNo         = ucSN;
        req.ucWriteDataLen  = ucCount;
        req.wStartAddr      = wStartPos;
        req.ucDataLen       = ucCount;

        dwError = USB_SendReq(&req);
        if (dwError) goto exit;
	}

exit:
    return (dwError);
}
//**********************************************************************
// Switch bus to tuner or from tuner
//
// Return:  0 if successful, non-zero otherwise
//**********************************************************************
static u32 USB_Switch2WireBus(
    u8 ucDemod2WireAddr,
    u8 ucSlaveDemod,
    u8 ucTuner             // 1: switch to tuner, 0: switch back
)
{
    u32      dwError = ERR_USB_NO_ERROR;
    u8       ucTemp;   

    if (ucSlaveDemod)
    {
        dwError = USB_ReadRegs(ucDemod2WireAddr, 1, p_reg_bypass_host2tuner, 1, &ucTemp);
        if (dwError) goto exit;

        ucTemp = REG_CREATE(ucTuner, ucTemp, reg_bypass_host2tuner_pos, reg_bypass_host2tuner_len, RegMasks);

        dwError = USB_WriteRegs(ucDemod2WireAddr, 1, p_reg_bypass_host2tuner, 1, &ucTemp);
        if (dwError) goto exit;
    }
    else
    {
        dwError = USB_ReadRegs(ucDemod2WireAddr, 0, I2C_reg_sel_tuner, 1, &ucTemp);
        if (dwError) goto exit;

        ucTemp = REG_CREATE(ucTuner, ucTemp, reg_sel_tuner_pos, reg_sel_tuner_len, RegMasks);

        dwError = USB_WriteRegs(ucDemod2WireAddr, 0, I2C_reg_sel_tuner, 1, &ucTemp);
        if (dwError) goto exit;
    }

exit:
    
    return (dwError);
}

u32 USB_WriteTunerRegs(
    u8 ucDemod2WireAddr,
    u8 ucSlaveDemod,
    u8 ucTuner2WireAddr,
    u8 ucTunerType,
    u16 wRegAddr,
    u8 ucCount,
    u8 * ucpBuf
)
{
    u32              dwError = ERR_USB_NO_ERROR;
    MercuryRequest      req;
    u8               ucRegAddrLen;

    dwError = USB_Switch2WireBus(ucDemod2WireAddr, ucSlaveDemod, 1);
    if (dwError) goto exit;

    if (ucTunerType == AF901X_SILICON_TUNER)
    {
        ucRegAddrLen = 1;
    }
    else if (ucTunerType == AF901X_CAN_TUNER)
    {
        ucRegAddrLen = 0;
    }
    else
    {
        dwError = ERR_TUNER_TYPE_NOT_SUPPORT;
        goto exit;
    }
    
    req.uc2WireAddr     = ucTuner2WireAddr;
    req.ucCmd           = USB_CMD_GENERAL_I2C;
    req.ucpReadDataPtr  = NULL;
    req.ucpWriteDataPtr = ucpBuf;
    req.ucReadDataLen   = 0;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = ucCount;
    req.wStartAddr      = wRegAddr;
    req.ucDataLen       = ucCount;
    req.ucRegAddrBytes  = ucRegAddrLen;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

	mdelay(1);

exit:

    if (dwError)
    {
        USB_Switch2WireBus(ucDemod2WireAddr, ucSlaveDemod, 0);
    }
    else
    {
        dwError = USB_Switch2WireBus( ucDemod2WireAddr, ucSlaveDemod, 0);
    }

    return (dwError);
}

u32 USB_ReadTunerRegs(
    u8 ucDemod2WireAddr,
    u8 ucSlaveDemod,
    u8 ucTuner2WireAddr,
    u8 ucTunerType,
    u16 wRegAddr,
    u8 ucCount,
    u8 * ucpBuf
)
{
    u32           dwError = ERR_USB_NO_ERROR;
    MercuryRequest   req;
    u8            ucRegAddrLen = 0;

    if (ucTunerType == AF901X_SILICON_TUNER)
    {
        ucRegAddrLen = 1;
    }
    
    //
    // Switch bus to tuner
    //
    dwError = USB_Switch2WireBus(ucDemod2WireAddr, ucSlaveDemod, 1);
    if (dwError) goto exit;

    req.uc2WireAddr     = ucTuner2WireAddr | 0x01;
    req.ucCmd           = USB_CMD_GENERAL_I2C;
    req.ucpReadDataPtr  = ucpBuf;
    req.ucpWriteDataPtr = NULL;
    req.ucReadDataLen   = ucCount;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = 0;
    req.wStartAddr      = wRegAddr;
    req.ucDataLen       = ucCount;
    req.ucRegAddrBytes  = ucRegAddrLen;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

	mdelay(1);

exit:

    if (dwError)
    {
        USB_Switch2WireBus(ucDemod2WireAddr, ucSlaveDemod, 0);
    }
    else
    {
        dwError = USB_Switch2WireBus(ucDemod2WireAddr, ucSlaveDemod, 0);
    }


    return (dwError);
}

u32 USB_ReDownloadFw(u8 check)
{
    u32 dwError = ERR_USB_NO_ERROR;
    MercuryRequest  req;

    req.uc2WireAddr     = 0;
    req.ucCmd           = USB_CMD_RE_DOWNLOAD_FW;
    req.ucpReadDataPtr  = NULL;
    req.ucpWriteDataPtr = NULL;
    req.ucReadDataLen   = 0;
    req.ucSeqNo         = ucSN;
    req.ucWriteDataLen  = 0;
    req.wStartAddr      = 0;
    req.ucDataLen       = 0;
    req.ucRegAddrBytes  = 0;

    dwError = USB_SendReq(&req);
    if (dwError) goto exit;

exit:
    return (dwError);
}
